{ Code to implement a 4th order Runge-Kutta integrator with adaptive step size
  control. This integrator is used to integrate the differential equations in
  the equations.pas file. }
unit integrator;

interface

uses sysutils, Dialogs, stypes, equations, fileio;

PROCEDURE odeIntegrator(VAR ystart: yValueArray; nvar: integer;
       x1,x2,dr,eps: double; var stepguess:double; stepmin: double;
       VAR numok,numbad: integer;
       var tdrive:drivearray; var tpar:paramarray);

implementation

uses frontend, options;

var
  kount:integer;
{  maxStepSave:integer;
  dxsave:double;
  xp: ARRAY [1..Maxstate] OF double;
  yp: ARRAY [1..10,1..Maxstate] OF double;  }

{ A fourth order runge kutta integrator. This function is called from
  RKQualityControl. }
PROCEDURE RungeKutta4(y, dydx: yValueArray; n: integer; x,xstep,dr: double;
    VAR yout: yValueArray; var tdrive:drivearray; var tpar:paramarray);
(* Programs using routine RungeKutta4 must provide a
PROCEDURE derivs(x:double; y:yValueArray; VAR dydx:yValueArray);
which returns the derivatives dydx at location x, given both x and the
function values y. The calling program must also define the types
TYPE
   yValueArray = ARRAY [1..nvar] OF double;
where nvar is the number of variables y. *)
VAR
   i: integer;
   xhalfstep,halfstep,xstep6: double;
   dym,dyt,ytemp: yValueArray;

BEGIN
  try
   halfstep := xstep*0.5;
   xstep6 := xstep/6.0;
   xhalfstep := x+halfstep;
   FOR i := 1 to n DO ytemp[i] := y[i] + halfstep*dydx[i];
   derivs(xhalfstep,dr,tdrive,tpar,ytemp,dyt);
   FOR i := 1 to n DO ytemp[i] := y[i] + halfstep*dyt[i];
   derivs(xhalfstep,dr,tdrive,tpar,ytemp,dym);
   FOR i := 1 to n DO
    BEGIN
      ytemp[i] := y[i] + xstep*dym[i];
      dym[i] := dyt[i]+dym[i];
    END;
   derivs(x+xstep,dr,tdrive,tpar,ytemp,dyt);
   FOR i := 1 to n DO yout[i] := y[i] + xstep6*(dydx[i]+dyt[i]+2.0*dym[i]);
  except
   raise;
  end;
END;

{ Step size control for the Runge Kutta integrator. }
PROCEDURE RKQualityControl(VAR y, dydx:yValueArray; n: integer; VAR x, dr: double;
      steptry,eps: double; yscale: yValueArray; VAR stepdid,stepnext: double;
      var tdrive:drivearray; var tpar:paramarray; var index: integer);
(* Programs using routine RKQualityControl must provide a
PROCEDURE derivs(x:double; y:yValueArray; VAR dydx:yValueArray);
which returns the derivatives dydx at location x, given both x and the
function values y. They must also define the type
TYPE
   yValueArray = ARRAY [1..n] OF double;
in the main routine.   *)
CONST
   pgrow=-0.20;
   pshrink=-0.25;          // pshrink=-0.1;         HERE
   fcor=0.06666666;   (* 1.0/15.0 *)
   one=1.0;
   safety=0.9;
   errcon=6.0e-4;
VAR
   i: integer;
   xsave,halfstep,step,temp,errmax: double;
   dydxsave,ysave,ytmp,ytemp,ytemp2: yValueArray;
//   errfile:textfile;
BEGIN
// assignfile(errfile,'errorfile.err');
// rewrite(errfile);
 xsave := x;
 ysave := y;
 dydxsave := dydx;
 step := steptry;
 errmax := 2;
//  try
 repeat
  try
    halfstep := 0.5*step;
    RungeKutta4(ysave,dydxsave,n,xsave,halfstep,dr,ytemp,tdrive,tpar);  // First half step
    x := xsave+halfstep;
    ytmp := ytemp;
    derivs(x,dr,tdrive,tpar,ytmp,dydx);
    RungeKutta4(ytemp,dydx,n,x,halfstep,dr,ytemp2,tdrive,tpar);     // Second half step
    y := ytemp2;
    x := xsave+step;
    IF (x = xsave) THEN
       begin
     raise EIntStepTooSmall.Create('Step size too small in routine RKQualityControl');
       end;
    RungeKutta4(ysave,dydxsave,n,xsave,step,dr,ytemp,tdrive,tpar);  // Large step
    errmax := 0.0;
    FOR i := 1 to n DO BEGIN         // Calculate error estimate
      ytemp[i] := y[i] - ytemp[i];
      temp := abs(ytemp[i]/yscale[i]);
      IF (errmax < temp) THEN
         begin
            errmax := temp;
            index := i;
//            writeln(errfile,stat[i].name:stringlength,' ',x,' ',y[i]:10,' ',ytemp[i]:10,' ',errmax:10,' ',safety*step*exp(pshrink*ln(errmax)):10);
         end;
    END;
    errmax := errmax/eps;
    IF (errmax > one) THEN
       step := safety*step*exp(pshrink*ln(errmax))
     ELSE BEGIN
       stepdid := step;
       IF (errmax > errcon) THEN
         stepnext := safety*step*exp(pgrow*ln(errmax))
       ELSE
         stepnext := 4.0*step;
     END;
  except
   on EZeroDivide do
    begin
     errmax := 2;
     step := 0.5*step;
    end;
  end;
 until errmax < one;
 FOR i := 1 to n DO y[i] := y[i] + ytemp[i]*fcor;
// finally closefile(errfile);
// end;
END;

{ Actual driver for the integrator. Prevents endless loops in the integrator
  and nonsignificant step sizes. }
PROCEDURE odeIntegrator(VAR ystart: yValueArray; nvar: integer;
       x1,x2,dr,eps: double; var stepguess:double; stepmin: double;
       VAR numok,numbad: integer;
       var tdrive:drivearray; var tpar:paramarray);
(* Programs using routine ODEINT must provide a
PROCEDURE derivs(x:double; y:yValueArray; VAR dydx:yValueArray);
which returns the derivatives dydx at location x, given both x
and the function values y. They must also define the type
TYPE
   yValueArray = ARRAY [1..nvar] OF double;
and declare the following parameter
VAR
   kount: integer;
in the main routine. *)
CONST
   maxstep=10000;
   two=2.0;
   zero=0.0;
   tiny=1.0e-30;
VAR
   numstep,i, index: integer;
   x,stepnext,stepdid,step:double;
   y,yscale,dydx: yValueArray;
BEGIN
   x := x1;
   IF (x2 > x1) THEN step := abs(stepguess) ELSE step := -abs(stepguess);
   numok := 0;
   numbad := 0;
   kount := 0;
   y := ystart;
   FOR numstep := 1 to maxstep DO
    BEGIN
     GetCurrentDrivers(dr,tdrive);
     derivs(x,dr,tdrive,tpar,y,dydx);
     FOR i := 1 to nvar DO
        begin
           if abs(y[i])> eps then yscale[i] := FmOptions.RunOptions.ErrorMult*
                                              (abs(y[i])+abs(dydx[i]*step)+tiny)
           else yscale[i] := FmOptions.RunOptions.ErrorMult*(abs(eps)+abs(dydx[i]*step)+tiny);
        end;
     IF (((x+step-x2)*(x+step-x1)) > zero) THEN step := x2-x;
     RKQualityControl(y,dydx,nvar,x,dr,step,eps,yscale,stepdid,stepnext,tdrive,tpar, index);
     IF (stepdid = step) THEN numok := numok+1
       ELSE numbad := numbad+1 ;
     IF (((x-x2)*(x2-x1)) >= zero){abs((x-x2))<epsilon {(((x-x2)*(x2-x1)) >= zero)} THEN        // HERE
      BEGIN
       ystart := y;
       stepguess := stepnext;
       exit;  // Normal exit
      END;
     IF (abs(stepnext) < stepmin) THEN
      BEGIN
       raise EIntStepTooSmall.Create('Step size too small in routine ODEIntegrator. '
         + 'Error caused by state variable, ' + stat[index].name + '.');
      END;
     step := stepnext;
   END;
   raise ETooManySteps.Create('Too many steps in routine ODEIntegrator');
   END;

end.
